/*****************************************************************************
 *
 * Copyright (c) ComponentOne, LLC.
 * Portions Copyright (c) 1999, KL GROUP INC.
 * http://www.componentone.com
 *
 * This file is provided for demonstration and educational uses only.
 * Permission to use, copy, modify and distribute this file for
 * any purpose and without fee is hereby granted, provided that the
 * above copyright notice and this permission notice appear in all
 * copies, and that the name of ComponentOne not be used in advertising
 * or publicity pertaining to this material without the specific,
 * prior written permission of an authorized representative of
 * ComponentOne.
 *
 * COMPONENTONE MAKES NO REPRESENTATIONS OR WARRANTIES ABOUT THE SUITABILITY
 * OF THE SOFTWARE, EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED
 * TO THE IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR
 * PURPOSE, OR NON-INFRINGEMENT. COMPONENTONE SHALL NOT BE LIABLE FOR ANY
 * DAMAGES SUFFERED BY USERS AS A RESULT OF USING, MODIFYING OR
 * DISTRIBUTING THIS SOFTWARE OR ITS DERIVATIVES.
 *
 *****************************************************************************/

/*
    This program is a simple simulation of the Stock Market.
    You start with a sum of money, and attempt to build a fortune.
    In terms of ComponentOne Chart features, it shows...

    - two charts on the same parent window
    - a Combination chart (stock price and portfolio value)
    - Fast-update routines to quickly update and scroll the display
*/

#include <windows.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <olch2d.h>
#include "stock.h"
#include "resource.h"

#define MAX(a, b)       ((a) > (b) ? (a) : (b))

// useful dollar amounts for the simulation
#define STARTING_CASH       ((double)500)
#define STARTING_SHARES     ((double)0)
#define BUYING_INCREMENT    ((double)100)
#define WIN_AMOUNT          ((double)2000)
#define LOSE_AMOUNT         ((double)100)
#define INITIAL_STARTING_PRICE  ((double)20)

#define SCROLL_WINDOW_SIZE  10
#define NUM_VOLUME_POINTS   3
#define MAX_DATA            50
#define HOLE_VALUE          XRT_HUGE_VAL
#define RND()               (rand()/(double)RAND_MAX)
#define TIMER_ID        1
#define TIME_OUT        10      // milliseconds
#define VCHART_HEIGHT   100
#define CTRLS_WIDTH     175

XrtDataHandle portfolioData = NULL;
XrtDataHandle priceData = NULL;
XrtDataHandle volData = NULL;

HWND hBuy, hSell, hGo, hStop, hRestart;
RECT rectShares, rectCash, rectPortfolio, rectStatus, rectRight;
int nVol;
double dPrice;
double dDay;

double dNumberOfSharess;
double dCashOnHand;
double dPortfolio;

HANDLE ghInst;
HWND gMainHwnd;
HXRT2D chartPrice;
HXRT2D chartVol;
HBRUSH hbrushBack;

BOOL bDisplayStatus = FALSE;
BOOL bRunning;

// AddData() creates new simulation data for a given "tick" of time
static void AddData(BOOL bUpdate)
{
    int bound;
    int frontDataset;
    int lastPoint;

    // is our scrolling window full?
    if (XrtDataGetLastPoint(priceData, 0) == (MAX_DATA-1)) {

        // shift old points in data to the left
        XrtArrDataShiftPts(portfolioData, 0, SCROLL_WINDOW_SIZE, MAX_DATA - SCROLL_WINDOW_SIZE);
        XrtDataSetLastPoint(portfolioData, 0, MAX_DATA - SCROLL_WINDOW_SIZE - 1);
        XrtArrDataShiftPts(priceData, 0, SCROLL_WINDOW_SIZE, MAX_DATA - SCROLL_WINDOW_SIZE);
        XrtDataSetLastPoint(priceData, 0, MAX_DATA - SCROLL_WINDOW_SIZE - 1);
        XrtArrDataShiftPts(volData, 0, NUM_VOLUME_POINTS * SCROLL_WINDOW_SIZE, NUM_VOLUME_POINTS * (MAX_DATA - SCROLL_WINDOW_SIZE));
        XrtDataSetLastPoint(volData, 0, NUM_VOLUME_POINTS * (MAX_DATA - SCROLL_WINDOW_SIZE - 1));

        // reset the X axis bounds
        XrtSetValues(chartPrice, XRT_XMIN, dDay - MAX_DATA + SCROLL_WINDOW_SIZE, XRT_XMAX, dDay + SCROLL_WINDOW_SIZE, NULL);
        XrtSetValues(chartVol, XRT_XMIN, dDay - MAX_DATA + SCROLL_WINDOW_SIZE, XRT_XMAX, dDay + SCROLL_WINDOW_SIZE, NULL);
    }

    /*
     *  Use sophisticated modeling analysis to determine an 
     *  appropriate price and volume data items.  Thus ensuring
     *  an accurate stock market simulation...
     */
    nVol = (int)(160*(1.1*RND() - RND()));
    dPrice += (double) sin((double)(nVol*RND()));
    dPrice = (double) fabs(dPrice);
    dPortfolio = dCashOnHand + (dPrice * dNumberOfSharess);

    // add new price data
    lastPoint = XrtDataGetLastPoint(priceData, 0);
    lastPoint++;
    XrtDataSetLastPoint(priceData, 0, lastPoint);
    XrtDataSetXElement(priceData, 0, lastPoint, dDay);
    XrtDataSetYElement(priceData, 0, lastPoint, dPrice);
    
    // add new portfolio data
    lastPoint = XrtDataGetLastPoint(portfolioData, 0);
    lastPoint++;
    XrtDataSetLastPoint(portfolioData, 0, lastPoint);
    XrtDataSetXElement(portfolioData, 0, lastPoint, dDay);
    XrtDataSetYElement(portfolioData, 0, lastPoint, dPortfolio);

    // add new volume data
    lastPoint = XrtDataGetLastPoint(volData, 0);
    lastPoint++;
    XrtDataSetLastPoint(volData, 0, lastPoint);
    XrtDataSetXElement(volData, 0, lastPoint, dDay);
    XrtDataSetYElement(volData, 0, lastPoint, HOLE_VALUE);
    lastPoint++;
    XrtDataSetLastPoint(volData, 0, lastPoint);
    XrtDataSetXElement(volData, 0, lastPoint, dDay);
    XrtDataSetYElement(volData, 0, lastPoint, 1000.f + 5*nVol);
    lastPoint++;
    XrtDataSetLastPoint(volData, 0, lastPoint);
    XrtDataSetXElement(volData, 0, lastPoint, dDay);
    XrtDataSetYElement(volData, 0, lastPoint, 0.0);

    // do we want to update the screen right now?
    if (bUpdate) {
        
        // check if the new data will fit without needing to scroll
        bound = XrtArrCheckAxisBounds(chartPrice, XRT_DATASET1, 1);
        
        // will new points fit into existing X axis bounds?
        if ((bound & XRT_GTX) == 0) {
            
            // find out which dataset should be in front
            XrtGetValues(chartPrice, XRT_FRONT_DATASET, &frontDataset, NULL);
            
            // update them in the appropriate order to ensure correct drawing
            if (frontDataset == XRT_DATASET1) {
                XrtArrDataFastUpdate(chartPrice, XRT_DATASET2, 1);
                XrtArrDataFastUpdate(chartPrice, XRT_DATASET1, 1);
            }
            else {
                XrtArrDataFastUpdate(chartPrice, XRT_DATASET1, 1);
                XrtArrDataFastUpdate(chartPrice, XRT_DATASET2, 1);
            }
            XrtArrDataFastUpdate(chartVol, XRT_DATASET1, 1);
        }
        else {
            // refresh the charts with new data
            XrtSetValues(chartPrice, XRT_DATA, priceData, NULL);
            XrtSetValues(chartPrice, XRT_DATA2, portfolioData, NULL);
            XrtSetValues(chartVol, XRT_DATA, volData, NULL);
        }
    }

    // move to the next day
    dDay += 1.0;
}


void DrawWorth()
{
    InvalidateRect(gMainHwnd, &rectStatus, FALSE);
}


// DrawWorth() updates the status area with current statistics
void PaintWorth(HDC hdc)
{
    COLORREF crForeground, crBackground;
    char szBuffer[100];

    // update turned off
    if (! bDisplayStatus) {
        return;
    }

    // find out what the current chart colors are
    XrtGetValues(chartPrice, XRT_GRAPH_FOREGROUND_COLOR, &crForeground,
        XRT_BACKGROUND_COLOR, &crBackground, NULL);
    
    // use them for updating the status area
    SetTextColor(hdc, crForeground);
    SetBkColor(hdc, crBackground);

    sprintf(szBuffer, "Shares: %.3f", dNumberOfSharess);
    ExtTextOut(hdc, rectShares.left, rectShares.top, ETO_CLIPPED | ETO_OPAQUE, 
        &rectShares, szBuffer, (int)strlen(szBuffer), NULL);

    sprintf(szBuffer, "Cash: $%.2f", dCashOnHand);
    ExtTextOut(hdc, rectCash.left, rectCash.top , ETO_CLIPPED | ETO_OPAQUE, 
        &rectCash, szBuffer, (int)strlen(szBuffer), NULL);

    sprintf(szBuffer, "Portfolio: $%.2f", dPortfolio);
    ExtTextOut(hdc, rectPortfolio.left, rectPortfolio.top, ETO_CLIPPED | ETO_OPAQUE, 
        &rectPortfolio, szBuffer, (int)strlen(szBuffer), NULL);
}


// ShowStatusWindows() toggles the display of the statistics
void ShowStatusWindows(BOOL bShow)
{
    int nCmdShow = bShow ? SW_SHOW : SW_HIDE; 

    if (bShow) {
        // if re-enabling statistics, make sure they are up to date
        DrawWorth();
    }

    else {
        HWND hwndPrice = XrtGetWindow(chartPrice);
        InvalidateRect(hwndPrice, &rectShares, FALSE);
        InvalidateRect(hwndPrice, &rectCash, FALSE);
        InvalidateRect(hwndPrice, &rectPortfolio, FALSE);
    }
}


// Dialog procedure used to display the rules
LRESULT CALLBACK RulesDlgProc(HWND hwnd, UINT message, WPARAM wParam, LPARAM lParam)
{
    HWND hwndChart;
    static HXRT2D hChart = NULL;
    static char *header[] = {"How To Play", NULL};
    static char *footer[] = {
        "Press the Go button to begin the simulation.",
        "The Buy and Sell buttons will buy and sell",
        "stock in $100 increments. The Stop button ",
        "will pause the simulation. The Restart button",
        "will restart the simulation.",
        "",
        "You begin with $500 in cash.",
        "You win when your portfolio value reaches $2,000.",
        "You lose when your portfolio value drops below $100.",
        NULL
    };

    switch (message) {
      case WM_INITDIALOG:
        hwndChart = GetDlgItem(hwnd, 101);
        hChart = XrtCreate();
        XrtAttachWindow(hChart, hwndChart);
        XrtSetValues(hChart,
            XRT_BORDER,                     XRT_BORDER_ETCHED_OUT,
            XRT_BORDER_WIDTH,               4,
            XRT_BACKGROUND_COLOR,           RGB(0xAA,0xAA,0xAA),
            XRT_HEADER_Y,                   10,
            XRT_HEADER_STRINGS,             header,
            XRT_FOOTER_BORDER,              XRT_BORDER_ETCHED_IN,
            XRT_FOOTER_BORDER_WIDTH,        2,
            XRT_FOOTER_BACKGROUND_COLOR,    RGB(0xCC,0xCC,0xCC),
            XRT_FOOTER_ADJUST,              XRT_ADJUST_LEFT,
            XRT_FOOTER_STRINGS,             footer,
            NULL);
        XrtSetPropString(hChart, XRT_FOOTER_FONT, "Arial,9");
        XrtSetPropString(hChart, XRT_HEADER_FONT, "Times,24");
        return(TRUE);
      case WM_COMMAND:
        EndDialog(hwnd, TRUE);
        XrtDestroy(hChart);
        hChart = NULL;
        return(TRUE);
    }
    return(FALSE);
}


void DisplayRules()
{
    DialogBox(ghInst, (LPCSTR)"Rules", gMainHwnd, (DLGPROC) RulesDlgProc);
}

// Dialog procedure used to display winner's message
LRESULT CALLBACK WinDlgProc(HWND hwnd, UINT message, WPARAM wParam, LPARAM lParam)
{
    HWND hwndChart;
    static HXRT2D hChart = NULL;
    static char *header[] = {"Congratulations!", "You *****WIN*****!", NULL};
    static char back_color[] = "PowderBlue";

    switch (message) {
      case WM_INITDIALOG:
        hwndChart = GetDlgItem(hwnd, 101);
        hChart = XrtCreate();
        XrtAttachWindow(hChart, hwndChart);
        XrtSetValues(hChart,
            XRT_BORDER,                     XRT_BORDER_ETCHED_OUT,
            XRT_BORDER_WIDTH,               2,
            XRT_BACKGROUND_COLOR,           RGB(0xAA,0xAA,0xAA),
            XRT_HEADER_STRINGS,             header,
            XRT_HEADER_BORDER,              XRT_BORDER_SHADOW,
            XRT_HEADER_BORDER_WIDTH,        5,
            NULL);
        XrtSetPropString(hChart, XRT_HEADER_FONT, "Times,16");
        XrtSetPropString(hChart, XRT_HEADER_BACKGROUND_COLOR, back_color);
        return(TRUE);
      case WM_COMMAND:
        EndDialog(hwnd, TRUE);
        XrtDestroy(hChart);
        hChart = NULL;
        return(TRUE);
    }
    return(FALSE);
}


void DisplayWin()
{
    DialogBox(ghInst, (LPCSTR)"Win", gMainHwnd, (DLGPROC) WinDlgProc);
}

// Dialog procedure used to display the loser's message
LRESULT CALLBACK LoseDlgProc(HWND hwnd, UINT message, WPARAM wParam, LPARAM lParam)
{
    HWND hwndChart;
    static HXRT2D hChart = NULL;
    static char *header[] = {"Sorry,", "you LOSE!", NULL};
    static char back_color[] = "RosyBrown";

    switch (message) {
      case WM_INITDIALOG:
        hwndChart = GetDlgItem(hwnd, 101);
        hChart = XrtCreate();
        XrtAttachWindow(hChart, hwndChart);
        XrtSetValues(hChart,
            XRT_BORDER,                     XRT_BORDER_ETCHED_OUT,
            XRT_BORDER_WIDTH,               2,
            XRT_BACKGROUND_COLOR,           RGB(0xAA,0xAA,0xAA),
            XRT_HEADER_STRINGS,             header,
            XRT_HEADER_BORDER,              XRT_BORDER_SHADOW,
            XRT_HEADER_BORDER_WIDTH,        5,
            NULL);
        XrtSetPropString(hChart, XRT_HEADER_FONT, "Times,16");
        XrtSetPropString(hChart, XRT_HEADER_BACKGROUND_COLOR, back_color);
        return(TRUE);
      case WM_COMMAND:
        EndDialog(hwnd, TRUE);
        XrtDestroy(hChart);
        hChart = NULL;
        return(TRUE);
    }
    return(FALSE);
}


void DisplayLose()
{
    DialogBox(ghInst, (LPCSTR)"Lose", gMainHwnd, (DLGPROC) LoseDlgProc);
}


void DisplayAbout(HWND hwnd)
{
    XrtAboutDialogBox(hwnd);
}


// Enable/disable Buy/Sell buttons while simulation is running
void EnableBuySellButtons()
{
    EnableWindow(hBuy, bRunning && (dCashOnHand >= BUYING_INCREMENT));
    EnableWindow(hSell, bRunning && (dNumberOfSharess >= (BUYING_INCREMENT / dPrice)));
}


// Enable/disable buttons depending on state of simulation
void EnableButtons()
{
    EnableBuySellButtons();

    EnableWindow(hGo, !bRunning);
    EnableWindow(hStop, bRunning);
    EnableWindow(hRestart, !bRunning);
}


// Try to buy a certain amount of shares
void BuyShares()
{
    // does he have enough to buy some shares?
    if (dCashOnHand >= BUYING_INCREMENT) {
        dCashOnHand -= BUYING_INCREMENT;
        dNumberOfSharess += (BUYING_INCREMENT / dPrice);
    }            
}


// Try to sell a certain amount of shares
void SellShares()
{
    // does he have enough to sell some shares?
    if (dNumberOfSharess >= (BUYING_INCREMENT / dPrice)) {
        dCashOnHand += BUYING_INCREMENT;
        dNumberOfSharess -= (BUYING_INCREMENT / dPrice);
    }
}


// Start the timer running for a simulation
void StartSimulation()
{
    SetTimer(gMainHwnd, TIMER_ID, TIME_OUT, NULL);
    bRunning = TRUE;
    EnableButtons();
}


// Stop the timer running for a simulation
void StopSimulation()
{
    KillTimer(gMainHwnd, TIMER_ID);
    bRunning = FALSE;
    EnableButtons();
}


// Determine whether the user has won/lost the game yet
BOOL CheckTermination()
{
    BOOL bStatus = FALSE;
    
    if ((dPortfolio >= WIN_AMOUNT) || (dPortfolio < LOSE_AMOUNT)) {
        StopSimulation();
        bStatus = TRUE;
    }
    
    if (dPortfolio >= WIN_AMOUNT) {
        DisplayWin();
    }

    else if (dPortfolio < LOSE_AMOUNT) {
        DisplayLose();
    }

    return(bStatus);
}


// Allocate space for data used in the simulation
void InitSimulation()
{
    // allocate enough space for the scrolling window of data
    portfolioData = XrtDataCreate(XRT_DATA_ARRAY, 1, MAX_DATA);
    priceData = XrtDataCreate(XRT_DATA_ARRAY, 1, MAX_DATA);
    volData = XrtDataCreate(XRT_DATA_ARRAY, 1, NUM_VOLUME_POINTS * MAX_DATA);

    // tell ComponentOne Chart about our memory allocation
    XrtSetValues(chartPrice, XRT_DATA2, portfolioData, NULL);
    XrtSetValues(chartPrice, XRT_DATA, priceData, NULL);
    XrtSetValues(chartVol, XRT_DATA, volData, NULL);
}


// Reset the simulation variables for a fresh start
void RestartSimulation()
{
    int i;

    dNumberOfSharess = STARTING_SHARES;
    dCashOnHand = STARTING_CASH;

    dPrice = INITIAL_STARTING_PRICE;
    dDay = 0.0;

    // we start with a (logically) empty window
    XrtDataSetLastPoint(portfolioData, 0, -1);
    XrtDataSetLastPoint(priceData, 0, -1);
    XrtDataSetLastPoint(volData, 0, -1);

    // reset the X axis
    XrtSetValues(chartPrice, XRT_XMIN, dDay, XRT_XMAX, dDay + MAX_DATA, NULL);
    XrtSetValues(chartVol, XRT_XMIN, dDay, XRT_XMAX, dDay + MAX_DATA, NULL);

    // disable painting of charts while we generate data
    XrtSetValues(chartPrice, XRT_REPAINT, FALSE, NULL);
    XrtSetValues(chartVol, XRT_REPAINT, FALSE, NULL);
    
    // fill up almost one window's worth of data
    for (i = 0; i < MAX_DATA-1; i++) {
        AddData(FALSE);
    }        
    
    // refresh the charts with new data
    XrtSetValues(chartPrice, XRT_DATA, priceData, NULL);
    XrtSetValues(chartPrice, XRT_DATA2, portfolioData, NULL);
    XrtSetValues(chartVol, XRT_DATA, volData, NULL);

    // re-enable painting of charts
    XrtSetValues(chartPrice, XRT_REPAINT, TRUE, NULL);
    XrtSetValues(chartVol, XRT_REPAINT, TRUE, NULL);

    // update statistics
    DrawWorth();

    // update buttons in user interface
    bRunning = FALSE;
    EnableButtons();
}


// Create all the controls used in the main windows
void CreateControls(HWND parent)
{
    /*
     *  Create the two charts and their controls on top of the price 
     *  chart.  Don't worry about the placement for now.  The size
     *  and position will be recalculated during the WM_SIZE message.
     */

    XrtDataStyle    *pDstyle;

    /*    WS_VISIBLE | WS_CHILD | WS_CLIPSIBLINGS, */
    chartPrice = XrtCreateWindow("price_props",
        0, 0, 0, 0, parent, ghInst);
    XrtSetValues(chartPrice,
        XRT_TYPE,                       XRT_TYPE_PLOT,
        XRT_TYPE2,                      XRT_TYPE_AREA,
        XRT_AXIS_BOUNDING_BOX,          TRUE,
        XRT_GRAPH_MARGIN_BOTTOM,        5,      // just over half the font-height
        XRT_BACKGROUND_COLOR,           RGB(0, 0x80, 0),
        XRT_FOREGROUND_COLOR,           RGB(0, 0, 0),
        XRT_GRAPH_FOREGROUND_COLOR,     RGB(0xFF, 0xFF, 0),
        XRT_DATA_AREA_BACKGROUND_COLOR, RGB(0,0,0),
        XRT_XANNOTATION_METHOD,         XRT_ANNO_TIME_LABELS,
        XRT_XAXIS_SHOW,                 FALSE,
        XRT_YTITLE,                     "Price",
        XRT_YTITLE_ROTATION,            XRT_ROTATE_90,
        XRT_YAXIS_MIN,                  0.0,
        XRT_YAXIS_MAX,                  100.0,
        XRT_YGRID_USE_DEFAULT,          TRUE,           // turn on grid lines
        XRT_Y2AXIS_MIN,                 0.0,
        XRT_Y2AXIS_MAX,                 WIN_AMOUNT,
        NULL);
    XrtSetPropString(chartPrice, XRT_AXIS_FONT, "Arial,8");

    // use solid grid lines
    XrtGetValues(chartPrice, XRT_YGRID_DATA_STYLE, &pDstyle, NULL);
    pDstyle->lpat = XRT_LPAT_SOLID;
    XrtSetValues(chartPrice, XRT_YGRID_DATA_STYLE, pDstyle, NULL);

    // use a solid, 2-pixel thick red line for the price
    pDstyle = XrtGetNthDataStyle(chartPrice, 0);
    pDstyle->lpat = XRT_LPAT_SOLID;
    pDstyle->width = 2;
    pDstyle->color = RGB(0xFF,0,0);
    pDstyle->point = XRT_POINT_NONE;
    XrtSetNthDataStyle(chartPrice, 0, pDstyle);

    // use a solid green area for the present value
    pDstyle = XrtGetNthDataStyle2(chartPrice, 0);
    pDstyle->lpat = XRT_LPAT_SOLID;
    pDstyle->color = RGB(0,0xFF,0);
    pDstyle->point = XRT_POINT_NONE;
    XrtSetNthDataStyle2(chartPrice, 0, pDstyle);

    chartVol = XrtCreateWindow("vol_props",
        0, 0, 0, 0, parent, ghInst);
    XrtSetValues(chartVol,
        XRT_TYPE,                       XRT_TYPE_PLOT,
        XRT_AXIS_BOUNDING_BOX,          TRUE,
        XRT_GRAPH_MARGIN_TOP,           0,
        XRT_BACKGROUND_COLOR,           RGB(0,0x80,0),
        XRT_FOREGROUND_COLOR,           RGB(0,0,0),
        XRT_GRAPH_FOREGROUND_COLOR,     RGB(0xFF,0xFF,0),
        XRT_HEADER_BACKGROUND_COLOR,    RGB(0xD2,0xB4,0x8C),
        XRT_DATA_AREA_BACKGROUND_COLOR, RGB(0xBE,0xBE,0xBE),
        XRT_XANNOTATION_METHOD,         XRT_ANNO_TIME_LABELS,
        XRT_TIME_BASE,                  0L,
        XRT_TIME_UNIT,                  XRT_TMUNIT_DAYS,
        XRT_YTITLE_ROTATION,            XRT_ROTATE_90,
        XRT_Y2TITLE_ROTATION,           XRT_ROTATE_90,
        XRT_YTITLE,                     "Volume",
        XRT_YGRID_USE_DEFAULT,          TRUE,
        XRT_YAXIS_MAX,                  2500.0,
        NULL);
    XrtSetPropString(chartVol, XRT_AXIS_FONT, "Arial,8");

    // use solid grid lines
    XrtGetValues(chartVol, XRT_YGRID_DATA_STYLE, &pDstyle, NULL);
    pDstyle->lpat = XRT_LPAT_SOLID;
    XrtSetValues(chartVol, XRT_YGRID_DATA_STYLE, pDstyle, NULL);

    // use solid blue for the volume
    pDstyle = XrtGetNthDataStyle(chartVol, 0);
    pDstyle->lpat = XRT_LPAT_SOLID;
    pDstyle->color = RGB(0,0,0xFF);
    pDstyle->point = XRT_POINT_NONE;
    XrtSetNthDataStyle(chartVol, 0, pDstyle);

    hBuy = CreateWindow("BUTTON", "Buy",
        WS_VISIBLE | WS_CHILD,
        0, 0, 70, 30, parent, NULL, ghInst, NULL);
    hSell = CreateWindow("BUTTON", "Sell",
        WS_VISIBLE | WS_CHILD,
        0, 0, 70, 30, parent, NULL, ghInst, NULL);
    hGo = CreateWindow("BUTTON", "Go",
        WS_VISIBLE | WS_CHILD,
        0, 0, 70, 30, parent, NULL, ghInst, NULL);
    hStop = CreateWindow("BUTTON", "Stop",
        WS_VISIBLE | WS_CHILD,
        0, 0, 70, 30, parent, NULL, ghInst, NULL);
    hRestart = CreateWindow("BUTTON", "Restart",
        WS_VISIBLE | WS_CHILD,
        0, 0, 70, 30, parent, NULL, ghInst, NULL);
}


// Resize/reposition the controls based on the new size of the main window
void SizeControls(int width, int height)
{
    int priceHeight;
    int volHeight;
    int chartWidth;
    int controlWidth;
    int controlHeight;
    int controlRight;
    int controlTop;
    int left_margin1, left_margin2;
    int right_margin1, right_margin2;

    priceHeight = height - VCHART_HEIGHT;
    volHeight = VCHART_HEIGHT;
    chartWidth = width - CTRLS_WIDTH;
    controlWidth = (CTRLS_WIDTH*8)/10;
    controlHeight = height/10;
    
    // place price in top portion of window
    SetWindowPos(XrtGetWindow(chartPrice), HWND_BOTTOM, 0, 0, chartWidth, priceHeight, 
        SWP_SHOWWINDOW | SWP_NOZORDER);

    // place volume in bottom portion of window
    SetWindowPos(XrtGetWindow(chartVol), HWND_BOTTOM, 0, priceHeight, chartWidth, volHeight, 
        SWP_SHOWWINDOW | SWP_NOZORDER);

    // place and resize buttons in right side of window
    SetWindowPos(hBuy, HWND_TOP, chartWidth, (int)((height*0.2)/10), 
         controlWidth, controlHeight, SWP_SHOWWINDOW | SWP_NOSIZE);
    SetWindowPos(hSell, HWND_TOP, chartWidth, (int)((height*1.4)/10), 
         controlWidth, controlHeight, SWP_SHOWWINDOW | SWP_NOSIZE);
    SetWindowPos(hGo, HWND_TOP, chartWidth, (int)((height*2.6)/10), 
         controlWidth, controlHeight, SWP_SHOWWINDOW | SWP_NOSIZE);
    SetWindowPos(hStop, HWND_TOP, chartWidth, (int)((height*3.8)/10), 
         controlWidth, controlHeight, SWP_SHOWWINDOW | SWP_NOSIZE);
    SetWindowPos(hRestart, HWND_TOP, chartWidth, (int)((height*5.0)/10), 
         controlWidth, controlHeight, SWP_SHOWWINDOW | SWP_NOSIZE);

    // place update text in bottom right side of window
    controlRight = chartWidth + CTRLS_WIDTH;

    controlTop = (int)((height*7.0)/10);
    SetRect(&rectShares, chartWidth, controlTop, controlRight, 
        controlTop + controlHeight);

    controlTop = (int)((height*8.0)/10);
    SetRect(&rectCash, chartWidth, controlTop, controlRight, 
        controlTop + controlHeight);

    controlTop = (int)((height*9.0)/10);
    SetRect(&rectPortfolio, chartWidth, controlTop, controlRight, 
        controlTop + controlHeight);

    SetRect(&rectStatus, chartWidth, (int)((height*6.0)/10), 
        controlRight, height);

    SetRect(&rectRight, chartWidth, 0, controlRight, height);

    // resize internal chart heights to match new size of window
    // reset margins to default, then calculate the maximum margins
    // of the two windows, and set them both to the same margin.  This
    // forces the annotation on the two windows to line up.
    XrtSetValues(chartPrice, 
        XRT_GRAPH_X,                        0,
        XRT_GRAPH_Y,                        0,
        XRT_GRAPH_WIDTH,                    chartWidth, 
        XRT_GRAPH_HEIGHT,                   priceHeight, 
        XRT_GRAPH_MARGIN_LEFT_USE_DEFAULT,  TRUE,
        XRT_GRAPH_MARGIN_RIGHT_USE_DEFAULT, TRUE,
        NULL);
    XrtSetValues(chartVol, 
        XRT_GRAPH_X,                        0,
        XRT_GRAPH_Y,                        0,
        XRT_GRAPH_WIDTH,                    chartWidth, 
        XRT_GRAPH_HEIGHT,                   volHeight, 
        XRT_GRAPH_MARGIN_LEFT_USE_DEFAULT,  TRUE,
        XRT_GRAPH_MARGIN_RIGHT_USE_DEFAULT, TRUE,
        NULL);


    XrtGetValues(chartPrice, XRT_GRAPH_MARGIN_LEFT, &left_margin1, XRT_GRAPH_MARGIN_RIGHT, &right_margin1, NULL);
    XrtGetValues(chartVol, XRT_GRAPH_MARGIN_LEFT, &left_margin2, XRT_GRAPH_MARGIN_RIGHT, &right_margin2, NULL);
    XrtSetValues(chartPrice,
            XRT_GRAPH_MARGIN_LEFT,          MAX(left_margin1, left_margin2),
            XRT_GRAPH_MARGIN_RIGHT,         MAX(right_margin1, right_margin2),
            NULL);
    XrtSetValues(chartVol,
            XRT_GRAPH_MARGIN_LEFT,          MAX(left_margin1, left_margin2),
            XRT_GRAPH_MARGIN_RIGHT,         MAX(right_margin1, right_margin2),
            NULL);

    InvalidateRect(gMainHwnd, &rectRight, TRUE);
}


// Main window WndProc
LRESULT CALLBACK MainWndProc(HWND hwnd, UINT msg, WPARAM wParam, LPARAM lParam)
{
    HWND    hwndCtrl;
    HDC hdc;
    PAINTSTRUCT ps;

    switch(msg) {
      case WM_COMMAND:
        
        // handle menu commands
        switch(LOWORD(wParam)) {
          case IDM_EXIT:
            DestroyWindow(hwnd);
            break;

          case IDM_RULES:
            DisplayRules();
            break;

          case IDM_STATUS:
            bDisplayStatus = !bDisplayStatus;
            CheckMenuItem(GetMenu(gMainHwnd), IDM_STATUS, 
                          bDisplayStatus ? MF_CHECKED : MF_UNCHECKED);
            DrawWorth();
            break;

          case IDM_PRINT:
            StopSimulation();
            PrintCharts();
            break;

		case IDM_ABOUTDEMO:
		case IDMHELP:
			WinHelp(gMainHwnd, "OLCH8DMO.HLP", HELP_CONTEXT, 21);
			break;

		case IDM_ABOUTC1CHART:
			WinHelp(gMainHwnd, "OLCH8DMO.HLP", HELP_CONTEXT, 19);
			break;

          // handle messages from push-buttons
          default:
            hwndCtrl = (HWND)lParam;
            if (hwndCtrl == hStop) {
                StopSimulation();
            }
            else if (hwndCtrl == hGo) {
                if (!CheckTermination()) {
                    StartSimulation();
                }
            }
            else if (hwndCtrl == hBuy) {
                BuyShares();
            }
            else if (hwndCtrl == hSell) {
                SellShares();
            }
            else if (hwndCtrl == hRestart) {
                RestartSimulation();
            }
            break;
        }
        return(0);

      case WM_TIMER:
        // update the data every timeout event
        if (wParam == TIMER_ID) {
            AddData(TRUE);
            DrawWorth();
            EnableBuySellButtons();
            CheckTermination();
        }
        return(0);
      
        /* 
         *  To properly handle colors the chart needs to be
         *  notified about any system palette changes.
         */
    case XRTN_PALETTECHANGED:
        SendMessage(XrtGetWindow(chartPrice), WM_QUERYNEWPALETTE, 0, 0);

        /* If the price chart didn't cause any system palette entries
           to be modified, a WM_PALETTECHANGED message may not have been
           sent to the volume chart.  Send the message here in case it
           was not sent yet.
         */
        SendMessage(XrtGetWindow(chartVol), WM_PALETTECHANGED, (WPARAM)hwnd, 0);
        break;

    case WM_QUERYNEWPALETTE:
        /* Make sure the window is at the top of the Z-order so that
           the background palettes can be realized before any other
           application gets a chance to realize a palette.
         */
        /* In Windows for Workgroups, this message can be received
           before the window is moved to the top of the Z-order
           causing the first WM_PALETTECHANGED message to be sent
           not to this window but the window which was "previously"
           at the top of the Z-order.
         */
        BringWindowToTop(hwnd);

        SendMessage(XrtGetWindow(chartPrice), WM_QUERYNEWPALETTE, 0, 0);

        /* If the price chart didn't cause any system palette entries
           to be modified, a WM_PALETTECHANGED message may not have been
           sent to the volume chart.  Send the message here in case it
           was not sent yet.
         */
        SendMessage(XrtGetWindow(chartVol), WM_PALETTECHANGED, (WPARAM)hwnd, 0);
        break;

    case WM_PALETTECHANGED:
        SendMessage(XrtGetWindow(chartPrice), msg, wParam, lParam);
        SendMessage(XrtGetWindow(chartVol), msg, wParam, lParam);
        break;

      case WM_SIZE: 
        // resize the charts and controls
        SizeControls((int) LOWORD(lParam), (int) HIWORD(lParam));
        return(0);
    
      case WM_ERASEBKGND:
        // we just need to erase the non-chart portion of the display
        FillRect((HDC) wParam, &rectRight, hbrushBack);
        return(TRUE);

      case WM_PAINT:
        hdc = BeginPaint(hwnd, &ps);
        PaintWorth(hdc);
        EndPaint(hwnd, &ps);
        break;

      case WM_DESTROY:
        PostQuitMessage(0);
        return(0);
    }

    return(DefWindowProc(hwnd, msg, wParam, lParam));
}


// Entry point for main program
int WINAPI WinMain(HINSTANCE hInstance, HINSTANCE hPrevInstance,
    LPSTR lpCmdLine, int nCmdShow)
{
    WNDCLASS    wc;
    MSG         msg;
    COLORREF    crBack;
	HACCEL      hAccel;

    if (!hPrevInstance) {
        memset(&wc, 0, sizeof(wc));
        wc.lpfnWndProc = (WNDPROC) MainWndProc;
        wc.hInstance = hInstance;
        wc.hIcon = LoadIcon(hInstance, "olch2dicon");
        wc.hCursor = LoadCursor(NULL, IDC_ARROW);
        wc.hbrBackground = GetStockObject(BLACK_BRUSH);
        wc.lpszMenuName = (LPSTR) "StocksMenu";
        wc.lpszClassName = (LPSTR) "Main";
    
        if (!RegisterClass(&wc)) {
            MessageBox(NULL, "Cannot RegisterClass()",
                       "Err! - TEST", MB_OK | MB_ICONEXCLAMATION);
            return(FALSE);
        }
    }

	hAccel = LoadAccelerators(hInstance, "StocksAccelerators");
    
	ghInst = hInstance;
    
    // create main window
    gMainHwnd = CreateWindow("Main", "Stock Simulation",
        WS_OVERLAPPEDWINDOW, 0, 0, 700, 400, NULL, NULL, ghInst, NULL);
    if (!gMainHwnd) {
        return 0;
    }

    // create all controls in main window
    CreateControls(gMainHwnd);

    // get the current background color used by the demo
    XrtGetValues(chartPrice, XRT_BACKGROUND_COLOR, &crBack, NULL);
    
    // use that color for the background of the rest of the window
    hbrushBack = CreateSolidBrush(crBack);

#if defined(_WIN64)
    SetClassLongPtr(gMainHwnd, GCLP_HBRBACKGROUND, (long) hbrushBack);
#elif defined(_WIN32)
    SetClassLong(gMainHwnd, GCL_HBRBACKGROUND, (long) hbrushBack);
#else
    SetClassWord(gMainHwnd, GCW_HBRBACKGROUND, (WORD) hbrushBack);
#endif

    InitSimulation();

    RestartSimulation();

    ShowWindow(gMainHwnd, nCmdShow);

    DrawWorth();

    DisplayRules();
    
    while (GetMessage(&msg, NULL, 0, 0)) {
		if (!TranslateAccelerator(gMainHwnd, hAccel, &msg)) { 
			TranslateMessage(&msg);
			DispatchMessage(&msg);
		}
    }

    // remember to free up memory
    XrtDataDestroy(priceData);
    XrtDataDestroy(volData);
    XrtDataDestroy(portfolioData);
    XrtDestroy(chartPrice);
    XrtDestroy(chartVol);

    DeleteObject(hbrushBack);

    return((int)msg.wParam);
}
